// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'package:meta/meta.dart';

import '../base/deferred_component.dart';
import '../base/logger.dart';
import '../convert.dart';
import '../dart/pub.dart';
import '../flutter_manifest.dart';
import '../project.dart';
import 'preview_manifest.dart';

/// Builds and manages the pubspec for the widget preview scaffold
class PreviewPubspecBuilder {
  const PreviewPubspecBuilder({
    required this.logger,
    required this.verbose,
    required this.offline,
    required this.rootProject,
    required this.previewManifest,
  });

  final Logger logger;
  final bool verbose;

  /// Set to true if pub should operate in offline mode.
  final bool offline;

  /// The Flutter project that contains widget previews.
  final FlutterProject rootProject;

  /// Details about the current state of the widget preview scaffold project.
  final PreviewManifest previewManifest;

  /// Adds dependencies on:
  ///   - dtd, which is used to connect to the Dart Tooling Daemon to establish communication
  ///     with other developer tools.
  ///   - flutter_lints, which is referenced by the analysis_options.yaml generated by the 'app'
  ///     template.
  ///   - google_fonts, which is used for the Roboto Mono font.
  ///   - stack_trace, which is used to generate terse stack traces for displaying errors thrown
  ///     by widgets being previewed.
  ///   - url_launcher, which is used to open a browser to the preview documentation.
  static const List<String> _kWidgetPreviewScaffoldDeps = <String>[
    'dtd',
    'flutter_lints',
    'google_fonts',
    'stack_trace',
    'url_launcher',
  ];

  /// Maps asset URIs to relative paths for the widget preview project to
  /// include.
  @visibleForTesting
  static Uri transformAssetUri(Uri uri) {
    // Assets provided by packages always start with 'packages' and do not
    // require their URIs to be updated.
    if (uri.path.startsWith('packages')) {
      return uri;
    }
    // Otherwise, the asset is contained within the root project and needs
    // to be referenced from the widget preview scaffold project's pubspec.
    return Uri(path: '../../${uri.path}');
  }

  @visibleForTesting
  static AssetsEntry transformAssetsEntry(AssetsEntry asset) {
    return AssetsEntry(
      uri: transformAssetUri(asset.uri),
      flavors: asset.flavors,
      transformers: asset.transformers,
    );
  }

  @visibleForTesting
  static FontAsset transformFontAsset(FontAsset asset) {
    return FontAsset(transformAssetUri(asset.assetUri), weight: asset.weight, style: asset.style);
  }

  @visibleForTesting
  static DeferredComponent transformDeferredComponent(DeferredComponent component) {
    return DeferredComponent(
      name: component.name,
      // TODO(bkonyi): verify these library paths are always package: paths from the parent project.
      libraries: component.libraries,
      assets: component.assets.map(transformAssetsEntry).toList(),
    );
  }

  Future<void> populatePreviewPubspec({required FlutterProject rootProject}) async {
    final FlutterProject widgetPreviewScaffoldProject = rootProject.widgetPreviewScaffoldProject;

    // Overwrite the pubspec for the preview scaffold project to include assets
    // from the root project.
    widgetPreviewScaffoldProject.replacePubspec(
      buildPubspec(
        rootManifest: rootProject.manifest,
        widgetPreviewManifest: widgetPreviewScaffoldProject.manifest,
      ),
    );

    // Adds a path dependency on the parent project so previews can be
    // imported directly into the preview scaffold.
    const String pubAdd = 'add';
    // Use `json.encode` to handle escapes correctly.
    final String pathDescriptor = json.encode(<String, Object?>{
      // `pub add` interprets relative paths relative to the current directory.
      'path': rootProject.directory.fileSystem.path.relative(rootProject.directory.path),
    });

    final PubOutputMode outputMode = verbose ? PubOutputMode.all : PubOutputMode.failuresOnly;
    await pub.interactively(
      <String>[
        pubAdd,
        if (offline) '--offline',
        '--directory',
        widgetPreviewScaffoldProject.directory.path,
        // Ensure the path using POSIX separators, otherwise the "path_not_posix" check will fail.
        '${rootProject.manifest.appName}:$pathDescriptor',
      ],
      context: PubContext.pubAdd,
      command: pubAdd,
      touchesPackageConfig: true,
      outputMode: outputMode,
    );

    // Adds dependencies required by the widget preview scaffolding.
    await pub.interactively(
      <String>[
        pubAdd,
        if (offline) '--offline',
        '--directory',
        widgetPreviewScaffoldProject.directory.path,
        ..._kWidgetPreviewScaffoldDeps,
      ],
      context: PubContext.pubAdd,
      command: pubAdd,
      touchesPackageConfig: true,
      outputMode: outputMode,
    );

    // Generate package_config.json.
    await pub.get(
      context: PubContext.create,
      project: widgetPreviewScaffoldProject,
      offline: offline,
      outputMode: outputMode,
    );

    previewManifest.updatePubspecHash();
  }

  void onPubspecChangeDetected() {
    // TODO(bkonyi): trigger hot reload or restart?
    logger.printStatus('Changes to pubspec.yaml detected.');
    populatePreviewPubspec(rootProject: rootProject);
  }

  @visibleForTesting
  FlutterManifest buildPubspec({
    required FlutterManifest rootManifest,
    required FlutterManifest widgetPreviewManifest,
  }) {
    final List<AssetsEntry> assets = rootManifest.assets.map(transformAssetsEntry).toList();

    final List<Font> fonts = <Font>[
      ...widgetPreviewManifest.fonts,
      ...rootManifest.fonts.map((Font font) {
        return Font(font.familyName, font.fontAssets.map(transformFontAsset).toList());
      }),
    ];

    final List<Uri> shaders = rootManifest.shaders.map(transformAssetUri).toList();

    final List<DeferredComponent>? deferredComponents =
        rootManifest.deferredComponents?.map(transformDeferredComponent).toList();

    return widgetPreviewManifest.copyWith(
      logger: logger,
      assets: assets,
      fonts: fonts,
      shaders: shaders,
      deferredComponents: deferredComponents,
    );
  }
}
